option nonotes;

************************************************************
** read in the data output by the multivar distrib macro  **
** for each replicate and stratum.                        **
************************************************************;
    
%macro runscorese(outpath,replicfirst,replicfirstse,repliclast,finaldataset,episodic,daily,wholefruit, beansandgreens, wholegrains, refinedgrains, 
dairy, sugars, satfat, sodium, kcal,
totalfruit, vegandlegumes, seafoodplant, allprotein, mufapufa, fruitjuice, nondarkgr, darkgr, legumes, seasoynut, mpeggs);

  /**************************************************************************/
  /** the macro runscorese provides a wrapper for the following steps.    ***/
  /*** STEP 9: SCORE EACH REPLICATE                                       ***/
  /*** Create ratios and score each one according to the relevant         ***/
  /*** standard                                                           ***/
  /***                                                                    ***/
  /*** STEP 10: CALCULATE STANDARD ERRORS USING BRR                       ***/
  /*** Use the BRR replicates to create standard errors for the mean and  ***/
  /*** the percentiles of interest                                        ***/
  /***                                                                    ***/
  /**************************************************************************/
  
/*****************************************************************************************/
/***           --- The paramters are  ---                                              ***/
/*** outpath           =  Folder where the distmc variables are and the location       ***/
/***                      for the data set to be saved                                 ***/
/*** replicfirst       =  The starting number for macro calls used to score the HEI.   ***/ 
/***                      Usually 0.                                                   ***/
/*** replicfirstse     =  The starting number for the macro calls used in calculating  ***/ 
/***                      the SEs for the scores. Usually 1.                           ***/
/*** repliclast        =  The ending number for the calls to the macros. The number    ***/
/***                      of BRR weights used in the analysis.                         ***/
/*** finaldataset      =  A valid SAS name for the final data set                      ***/
/*** episodic          =  A list of episodic variable names - separated by spaces      ***/
/*** daily             =  A list of non-episodice variable names - separated by spaces ***/
/*** wholefruit        =  A valid SAS variable name to be output for wholefruit        ***/
/*** beansandgreens    =  A valid SAS variable name to be output for beansandgreens    ***/
/*** wholegrains       =  A valid SAS variable name to be output for wholegrains       ***/
/*** refinedgrains     =  A valid SAS variable name to be output for refinedgrains     ***/
/*** dairy             =  A valid SAS variable name to be output for dairy             ***/
/*** sugars            =  A valid SAS variable name to be output for sugars            ***/
/*** satfat            =  A valid SAS variable name to be output for satfat            ***/
/*** sodium            =  A valid SAS variable name to be output for sodium            ***/
/*** kcal              =  A valid SAS variable name to be output for kcal              ***/
/*** totalfruit        =  A valid SAS variable name to be output for totalfruit        ***/             
/*** vegandlegumes     =  A valid SAS variable name to be output for vegandlegumes     ***/
/*** seafoodplant      =  A valid SAS variable name to be output for seafoodplant      ***/
/*** allprotein        =  A valid SAS variable name to be output for all protein       ***/
/*** mufapufa          =  A valid SAS variable name to be output for mufapufa          ***/
/*** fruitjuice        =  A valid SAS variable name to be output for fruitjuice        ***/
/*** nondarkgr         =  A valid SAS variable name to be output for nondarkgr         ***/
/*** darkgr            =  A valid SAS variable name to be output for darkgr            ***/
/*** legumes           =  A valid SAS variable name to be output for legumes           ***/
/*** seasoynut         =  A valid SAS variable name to be output for seasoynut         ***/
/*** mpeggs            =  A valid SAS variable name to be output for mpeggs            ***/
/*****************************************************************************************/         
   ;
  /*************************************************************************************/;
  
  %macro runscore(outpath,replicfirst,repliclast,finaldataset,episodic,daily,wholefruit, beansandgreens, wholegrains, refinedgrains,
  dairy, sugars, satfat, sodium, kcal,
  totalfruit, vegandlegumes, seafoodplant, allprotein, mufapufa, fruitjuice, nondarkgr, darkgr, legumes, seasoynut, mpeggs);
  
  /*******************************************************************************/
  /*** the macro runscore performs STEP 9 - SCORE EACH REPLICATE.              ***/
  /*** The paramaters are a subset of those described for the macro runscorese ***/
  /*******************************************************************************/
 
  title%eval(&titles+2) "STEP 9: SCORE EACH REPLICATE" ; run;
  
  %put starting STEP 9: SCORE EACH REPLICATE;
  
  libname outlib "&liblabel" ; 
   
  /*************************************************************/
  /** create a macro variable "list"that includes only        **/
  /** the food variables to be used in this execution         **/
  /** to create names of variables in macro variable equality **/
  /*************************************************************/ 

  %let list=&episodic &daily;
  
  ** Count the number of words in the macro variable list **;
  %local count;
  %let count=0; 
  %do %while(%qscan(&list,&count+1,%str( )) ne %str());
    %let count = %eval(&count+1); 
  %end;
  
  data _null_;
  format k $32. equality $450.;
  indlist='';
  %do i = 1 %to &count;
        k="%scan(&list,&i)";
  		  equality=trim(equality)||'usintake_'||strip(k)||"= mc_t&i.;"; 
  %end;
  call symput('equalities',equality);
  drop i;
  run;
  
  %put &equalities;
  
  
  %do replicnum = &replicfirst %to &repliclast;
  
    **********************************************************************;
    ** read in all the data sets output from the multivar_distrib macro. **
    ** This example shows 3 strata. You should adapt the code to read in **
    ** the data sets for each strata in your analysis.                   **
    ***********************************************************************;
    
    data usintake_mc_t (drop=mc_t1-mc_t15);
    set outlib.distmc1&replicnum outlib.distmc2&replicnum outlib.distmc3&replicnum;
          
    ***********************************************************************
    *** Divide weights by set_number_monte_carlo_rand_obs value used in ***
    *** the MULTIVAR_DISTRIB macro.                                     ***
    ***********************************************************************;
      
    weight_nw_sumw_div100 = weight_nw_sumw / 100;  **should not be needed -- weights do not change but prevents overflow error;
     
    ********************************************************
    *** Assign variable names for usual intakes from the ***
    *** MULTIVAR_DISTRIB macro.                          ***
    ********************************************************;
      
      &equalities;
          
      usintake_&totalfruit     = usintake_&fruitjuice + usintake_&wholefruit;    
      usintake_&vegandlegumes  = usintake_&nondarkgr + usintake_&darkgr + usintake_&legumes;
      usintake_&beansandgreens = usintake_&darkgr + usintake_&legumes;
      usintake_&seafoodplant   = usintake_&seasoynut + (usintake_&legumes * 4);
      usintake_&allprotein     = usintake_&mpeggs + usintake_&seasoynut  + (usintake_&legumes * 4);
          
      output usintake_mc_t;
    
    run;
    
    ******************************************************************** 
    *** Calculate usual HEI-2015 scores using the multivariate Monte ***
    *** Carlo distribution of usual intakes and a SAS macro,         ***
    *** hei2010.score.macro.sas                                      ***
    *** (see http://riskfactor.cancer.gov/tools/hei/).               ***
    ******************************************************************** ;
    
        %hei2015(indat            = usintake_mc_t,
                 kcal             = usintake_&kcal,
      	         vtotalleg        = usintake_&vegandlegumes,
      	         vdrkgrleg        = usintake_&beansandgreens,
      	         f_total          = usintake_&totalfruit,
      	         fwholefrt        = usintake_&wholefruit,
      	         g_whole          = usintake_&wholegrains,
      	         d_total          = usintake_&dairy,
                 pfallprotleg     = usintake_&allprotein,
      	         pfseaplantleg    = usintake_&seafoodplant,
      	         monopoly         = usintake_&mufapufa,
      	         satfat           = usintake_&satfat,
      	         sodium           = usintake_&sodium,
      	         g_refined        = usintake_&refinedgrains,
      	         add_sugars       = usintake_&sugars, 
                 outdat           = usintake_dens_hei  
                 );
    
    *********************************************************************
    *** Calculate mean usual HEI-2015 scores and selected percentiles ***
    *********************************************************************;
    
    proc datasets nolist; delete hei_stat; run;
    
    %macro getmean(cvar,name);
    
    proc means data=usintake_dens_hei noprint;
      weight weight_nw_sumw_div100;
      var    &cvar; 
      output out  = hei_&cvar   mean = mean_score  
                 p1   = p1         
                 p5   = p5         
                 p10  = p10         
                 p25  = p25         
                 p50  = p50         
                 p75  = p75         
                 p90  = p90         
                 p95  = p95         
                 p99  = p99;           
        run;
    
    data hei_&cvar; format component $32. replicate 8.0; set hei_&cvar; component="&name"; replicate=&replicnum;run;
          
    ****************************************
    ** combine all the HEI stat data sets **
    ****************************************;
        
    proc append base=hei_stat data=hei_&cvar;  run;
    
    proc datasets nolist; delete hei_&cvar; run;
    
    %mend getmean;

   
    %let cvarlist=HEI2015C1_TOTALVEG HEI2015C2_GREEN_AND_BEAN HEI2015C3_TOTALFRUIT HEI2015C4_WHOLEFRUIT HEI2015C5_WHOLEGRAIN HEI2015C6_TOTALDAIRY HEI2015C7_TOTPROT HEI2015C8_SEAPLANT_PROT
    HEI2015C9_FATTYACID HEI2015C10_SODIUM HEI2015C11_REFINEDGRAIN HEI2015C12_SFAT HEI2015C13_ADDSUG hei2015_total_score usintake_&kcal;
    
    %let namelist=&vegandlegumes &beansandgreens &totalfruit &wholefruit &wholegrains &dairy &allprotein &seafoodplant &mufapufa._&satfat. &sodium &refinedgrains &satfat &sugars total &kcal;
    
    %local counth;
    %let counth=0; 
    %do %while(%qscan(&cvarlist,&counth+1,%str( )) ne %str());
    %let counth = %eval(&counth+1); 
    %end;
    
    %put Replicate &replicnum  COUNT OF COMPONENTS + TOTALs should be 15: FOUND  &counth;
    
    ***********************
    ** call the macro    **
    ***********************;
    %do i = 1 %to &counth;
      %getmean(cvar=%scan(&cvarlist,&i), name=%scan(&namelist,&i));
    %end;
    
    
        ****************************************
        ** review the HEI statistical data    **
        ****************************************;
         
        proc print data=hei_stat;
          title%eval(&titles+3) " HEI Score Statistics for Replicate Number &replicnum";
        run;   
        
        data hei&replicnum;
          set hei_stat;
          repl=&replicnum;
          run; 
      
        %if &replicnum=&replicfirst %then %do;
           data result;
             set hei&replicnum;
           run;
        %end;
        %else %do ;
          proc append base=result data=hei&replicnum;
        %end;
    
      
           
      *******************************************************************
      *** end of the HEI scores macro and subsequent data processing  ***
      *******************************************************************;  
        
     
          
    %end;   ** of replicate loops **;
  
  
  
  *******************************************************;
  *** create a final HEI data set from all replicates ***;
  *******************************************************; 
      data outlib.&finaldataset;  **total (all ages) hei stats (mean and percentages)**;
        set result;
        if repl > .;
      run;  

  %mend runscore;  
  
  /*proc datasets nolist; delete results; run; */
  
  
  /*****************************************************/
  /** call the runscore macro to calculate HEI scores **/
  /*****************************************************/
  ;
  
  %runscore(outpath="&liblabel", replicfirst=&replicfirst,repliclast=&repliclast, finaldataset=&finaldataset,
  episodic=&episodic, daily=&daily,
  wholefruit=&wholefruit, beansandgreens=&beansandgreens, wholegrains=&wholegrains, refinedgrains=&refinedgrains, dairy=&dairy, sugars=&sugars, satfat=&satfat, sodium=&sodium, 
  kcal=&kcal, totalfruit=&totalfruit, vegandlegumes=&vegandlegumes, seafoodplant=&seafoodplant, allprotein=&allprotein, mufapufa=&mufapufa, fruitjuice=&fruitjuice, nondarkgr=&nondarkgr,
  darkgr=&darkgr, legumes=&legumes, seasoynut=&seasoynut, mpeggs=&mpeggs);
  
 
  /*******************************************/
  /*** STEP 10: CALCULATE STANDARD ERRORS  ***/
  /*******************************************/
  ;
  
  title%eval(&titles+2) "STEP 10: CALCULATE STANDARD ERRORS USING BRR "; run;

  %put starting STEP 10: CALCULATE STANDARD ERRORS USING BRR;

  
  %macro brr(replicfirst,repliclast,finaldataset,totalfruit, wholefruit, vegandlegumes, beansandgreens, wholegrains,  dairy, allprotein, seafoodplant, mufapufa, satfat, 
  refinedgrains, sodium, sugars, kcal);
    /**************************************************************/
    /*** the macro brr calculates standard errors of HEI scores ***/
    /*** beween the survey and BRR weights                      ***/
    /**************************************************************/
    ;
    
    proc datasets nolist; delete brrreps; run;

	ods listing close;
    ods pdf file="&outpath.hei_plots.pdf";
    
    %let comps=&totalfruit &wholefruit &vegandlegumes &beansandgreens &wholegrains &dairy &allprotein &seafoodplant &mufapufa._&satfat &refinedgrains &sodium &satfat &sugars &satfat total &kcal ;
    
    %local count;
    %let count=0; 
    %do %while(%qscan(&comps,&count+1,%str( )) ne %str());
    %let count = %eval(&count+1); 
    %end;
  
	%do i=1 %to &count;
    
    %let k=%scan(&comps,&i);
    
    data _null_;
    set outlib.&finaldataset;
    if replicate=0;
    if component="&k";
    call symput('comp0',mean_score);
    call symput('p1',p1);
    call symput('p99',p99);
    run;
    
    data brrs;
    set outlib.&finaldataset;
    if replicate>0;
    if component="&k";
    run;
     
    %if "&k"="&kcal" %then %let by=100; %else %if "&k"="total" %then %let by=10; 
      %else %if "&k"="PFALLPROTLEG" %then %let by=.1 ;%else %let by=1;
          
    title%eval(&titles+4) "&k";
    proc sgplot data=brrs;
    scatter y=mean_score x=replicate;
    refline &comp0 /axis=y label="Baseline Run" lineattrs=(color=gray thickness=2) ;
    yaxis  label="&k" values=(&p1 to &p99 by &by) refticks;
    run;
    
    %end;

	ods pdf close ;  
    ods listing;

    
    data rep0;
    set outlib.&finaldataset;
    if replicate=0;
    run;
    proc sort; by component;
    
    %do replicnum = &replicfirst %to &repliclast;
    
    data brrrep;
    set outlib.&finaldataset;
    if replicate=&replicnum;
    run;
    proc sort; by component;
    
    
    ***********************************************;
    *** Merge with dataset with point estimates ***;
    *** and calculate sums of squares           ***;
    ***********************************************;
    
    data ss;
      merge rep0 brrrep(rename=(mean_score=mean_scoreb p1=p1b p5=p5b p10=p10b p25=p25b p50=p50b p75=p75b p90=p90b p95=p95b p99=p99b)); by component;
      array bvar (*) mean_scoreb p1b p5b p10b p25b p50b p75b p90b p95b p99b;
      array varo (*) mean_score  p1  p5  p10  p25  p50  p75  p90  p95  p99;
      array dsqr (*) mean_scores p1s p5s p10s p25s p50s p75s p90s p95s p99s;
      do i=1 to dim(bvar);
       dsqr[i]=(bvar[i]-varo[i])**2;
      end;
      keep component replicate mean_scores p1s p5s p10s p25s p50s p75s p90s p95s p99s;
    run;
    
    proc append base=brrreps data=ss; run;
    
    proc datasets nolist; delete brrrep ss; run;
    
    %end;
    
    *********************;
    *** Calculate Sum ***;
    *********************;
    
    proc sort data=brrreps; by component replicate;
    
    proc means data=brrreps sum noprint; by component;
      var  mean_scores p1s p5s p10s p25s p50s p75s p90s p95s p99s;
     output out=sums sum=mean_scoress p1ss p5ss p10ss p25ss p50ss p75ss p90ss p95ss p99ss;
    run;
    
    *********************;
    *** Calculate SE  ***;
    *********************;
    
    data brr;
     set sums;
     array sumt (*) mean_scoress p1ss p5ss p10ss p25ss p50ss p75ss p90ss p95ss p99ss;
     array se  (*)  mean_score p1 p5 p10 p25 p50 p75 p90 p95 p99;
       do j=1 to dim(sumt);
       se[j]=sqrt((sumt[j])/(16 * .49));
     end;
     keep component mean_score p1 p5 p10 p25 p50 p75 p90 p95 p99;
    run;
    
    ************************************************************;
    ***  Append SE to Point Estimates                         **;
    **   Save the HEI data set with the additional statistics **;  
    **  Create a work data set for use in the reports.        **;
    ************************************************************;
    
    data outlib.tdheistat16SE ;
      merge rep0 brr(rename=(mean_score=mean_score_se p1=p1_se p5=p5_se p10=p10_se p25=p25_se p50=p50_se p75=p75_se p90=p90_se p95=p95_se p99=p99_se)); by component; 
      
      component=upcase(component);
      label component='HEI Component'
	        lcl_95='Lower 95% CI Limit'
    			mean_score='Mean'
    			mean_score_se='SE Mean'
    			p1='1st Percentile'
    			p5='5th Percentile'
    			p10='10th Percentile'
    			p25='25th Percentile'
    			p50='50th Percentile'
    			p75='75th Percentile'
    			p90='90th Percentile'
    			p95='95th Percentile'
    			p99='99th Percentile'
    			p1_se='SE 1st Percentile'
    			p5_se='SE 5th Percentile'
    			p10_se='SE 10th Percentile'
    			p25_se='SE 25th Percentile'
    			p50_se='SE 50th Percentile'
    			p75_se='SE 75th Percentile'
    			p90_se='SE 90th Percentile'
    			p95_se='SE 95th Percentile'
    			p99_se='SE 99th Percentile'
    			pvalue='p-value'
    			ucl_95='Upper 95% CI Limit'
    			waldchisq='Wald chi square';

      lcl_95=mean_score-(2*mean_score_se);
      ucl_95=mean_score+(2*mean_score_se);
      waldchisq = (mean_score/mean_score_se)**2;
      pvalue = 1 - probchi(waldchisq,1);
      
      keep component mean_score p1 p5 p10 p25 p50 p75 p90 p95 p99 mean_score_se p1_se p5_se p10_se p25_se p50_se p75_se p90_se p95_se p99_se lcl_95 ucl_95 waldchisq pvalue;
    run;
    
    
    Proc contents data= outlib.tdheistat16SE ;
      title%eval(&titles+2) "Description of the data set with HEI Scores and Statistics";
    run;    

    ** Write a report showing HEI mean scores, selected percentiles and SEs **;
      
    proc format  ;
      value pvalf
       0-<.0001 = '<.0001' ;
       value $compf
      'F_TOTAL'='Total Fruits'
      'FWHOLEFRT'='Whole Fruits'
      'VTOTALLEG'='Total Vegetables'
      'VDRKGRLEG'='Greens and Beans'
      'G_WHOLE'='Whole Grains'
      'D_TOTAL'='Dairy'
      'PFALLPROTLEG'='Total Protein Foods'
      'PFSEAPLANTLEG'='Seafood and Plant Proteins'
      'G_REFINED'='Refined Grains'
      'MONOPOLY_TSFAT' = 'Fatty Acids'
      'TSODI'='Sodium'
      'TSFAT'='Saturated Fats'
      'ADD_SUGARS'='Added Sugars'
      'TOTAL'='Total Score'
      'TKCAL'='Total Energy'
      ;
     value $color
      'Percentiles' = 'lightgrey'
      ;
    run;
       
      data report_set;
      	set outlib.tdheistat16SE;
      	
        length mean_score_c $20 lcl_95_c ucl_95_c $8. ;        
        component=upcase(component);
        
      	array scorear{*} mean_score p1 p5 p10 p25 p50 p75 p90 p95 p99;
        array sear{*} mean_score_se p1_se p5_se p10_se p25_se p50_se p75_se p90_se p95_se p99_se;
        array scorech{*} $ mean_score_c p1_c p5_c p10_c p25_c p50_c p75_c p90_c p95_c p99_c;
      
      do i = 1 to dim(sear) ;
        sear(i)= -1*sear(i) ;  /* recode Ses to negative then format with negaparen to get the formatting for the table */
        if sear(i) = 0 then sear(i) = -0.00000000001 ;
        if component eq 'TKCAL' then scorech(i) = (put(scorear(i),7.))||' '||(put(sear(i),negparen7.1));
        else scorech(i) = (put(scorear(i),7.2))||' '||(put(sear(i),negparen7.2));
      end;
      
      if component eq 'TKCAL' then do;
        lcl_95_c = put((round(lcl_95,1)),8.0);
        ucl_95_c = put((round(ucl_95,1)),8.0);
      end;
      else do;
        lcl_95_c = put(lcl_95,8.2);
        ucl_95_c = put(ucl_95,8.2);
      end;      
      
        select(component);
        when('F_TOTAL') n_order = 1;
        when('FWHOLEFRT') n_order = 2;
        when('VTOTALLEG') n_order = 3;
        when('VDRKGRLEG') n_order = 4;
        when('G_WHOLE') n_order = 5;
        when('D_TOTAL') n_order = 6;
        when('PFALLPROTLEG') n_order = 7;
        when('PFSEAPLANTLEG') n_order = 8;
        when('MONOPOLY_TSFAT') n_order = 9;
        when('G_REFINED') n_order = 10;
        when('TSODI') n_order = 11;
        when('ADD_SUGARS') n_order = 12;
        when('TSFAT') n_order = 13;        
        when('TOTAL') n_order = 14;
        when('TKCAL') n_order = 15;
        otherwise;
        end;
        if n_order;
      
      run;
      
      proc sort data=report_set;
      	by n_order;
      run;
      

    *****************************************************
    ** send the report to a .pdf named hei_scores_SE.  **
    *****************************************************;
    
    ods pdf file="&outpath.hei_scores_SE.pdf";
    
      OPTIONS ORIENTATION=LANDSCAPE center nodate nonumber;
      ods escapechar='^';
      ods pdf file="&outpath.hei_scores_SE.pdf" style=journal;
      title%eval(&titles+1) j=l "How to Fit an MCMC Model & Calculate HEI Scores Using NHANES Data and BRR Weights";
      title%eval(&titles+2) j=l "STEP 10: CALCULATE STANDARD ERRORS USING BRR";
      title%eval(&titles+3) "HEI Scores";
      proc report data=report_set nowd missing
      	style(header) = { background = $color.}
      	;
      	columns ('Mean HEI Score' (component mean_score_c))
                   ('Percentiles' (p1_c p5_c p10_c p25_c p50_c p75_c p90_c p95_c p99_c))
                   ('95% CI' (lcl_95_c ucl_95_c)) ('P-Value' (waldchisq pvalue));
        define component/display "HEI Component" format=$compf. style(header)={just=left};
        define  mean_score_c/display "Score (se)" right;
        define  p1_c/display "p1" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p5_c/display "p5" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p10_c/display "p10" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p25_c/display "p25" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p50_c/display "p50" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p75_c/display "p75" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p90_c/display "p90" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p95_c/display "p95" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        define  p99_c/display "p99" right style(header)={background=lightgrey} style(column) = { background = lightgrey};
        /*define lcl_95 /display "Lower" flow format=8.2;
        define ucl_95 /display "Upper" flow  format=8.2;*/
        define lcl_95_c/display "Lower" right ;
        define ucl_95_c/display "Upper" right ;
        define waldchisq/display "Wald Chi-square" flow format=8.2;
        define pvalue/display "P-value" format=pvalue6.4;
        
      run;
      ods pdf close;         
    
  %mend brr;
  

  %brr(replicfirst=&replicfirstse,repliclast=&repliclast,finaldataset=&finaldataset,
  totalfruit=&totalfruit, wholefruit=&wholefruit, vegandlegumes=&vegandlegumes, beansandgreens=&beansandgreens, wholegrains=&wholegrains, dairy=&dairy, allprotein=&allprotein,
  seafoodplant=&seafoodplant, mufapufa=&mufapufa, satfat=&satfat, refinedgrains=&refinedgrains, sodium=&sodium, sugars=&sugars, kcal=&kcal)
  
 %put Done!;
  
  %mend runscorese;
  
